/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY                          *
 *                                                                            *
 * This program is free software; you can redistribute it and/or modify       *
 * it under the terms of the GNU General Public Liense as published by        *
 * the Free Software Foundation, either version 2 of the License, or (at      * 
 * your option) any later version.                                            *
 *                                                                            *
 * The ITX package is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY, without even the implied warranty of MERCHANTABILITY *
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
 * for more details.                                                          * 
 *                                                                            *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.                                                   *
 *                                                                            * 
 * Contact information:                                                       *
 * Donna Bergmark                                                             *
 * 484 Rhodes Hall                                                            *
 * Cornell University                                                         *
 * Ithaca, NY 14853-3801                                                      *
 * bergmark@cs.cornell.edu                                                    *
 ******************************************************************************/
package shared;

import java.io.*;

/**
 * Describes a SPOT presentation and contains helper functions for constructing
 * the various file names of the presentation components.
 * 
 * @version 1.0, 3/24/1999
 * @author Jason Howes
 */
public class PresentationInfo implements Serializable
{	
	/**
	 * Current directory String
	 */
	public static final String CURRENT_DIRECTORY = new String(".");
	
	/**
	 * The presentation path
	 */
	protected String mPath;

	/**
	 * Class constructor.
	 * 
	 * @param presentatonPath the SPOT presentation path
	 */
	public PresentationInfo(String presentationPath)
	{
		int firstSeparator;
		int lastSeparator;
		String newPath = new String(presentationPath);
		
		// Remove leading file separator from the presentation path
		firstSeparator = newPath.indexOf(File.separatorChar);
		if (firstSeparator == 0)
		{
			newPath = newPath.substring(1);
		}
		
		// Set the new presentationPath
		mPath = newPath;
	}
	
	/**
	 * Class constructor
	 * 
	 * @param SPOTFile a SPOT file associated with the presentation
	 * @param fileSuffix the associted file suffix
	 */
	public PresentationInfo(String SPOTFile, String fileSuffix)
	{
		String newPath = new String(SPOTFile);
		int firstSeparator;
		int lastSeparator;
		
		// Is the name a valid SPOT filename?
		if (newPath.length() > fileSuffix.length())
		{
			newPath = newPath.substring(0, newPath.length() - fileSuffix.length());			
		}
		
		// Remove leading file separator from the presentation path
		firstSeparator = newPath.indexOf(File.separatorChar);
		if (firstSeparator == 0)
		{
			newPath = newPath.substring(1);
		}		
		
		// Set the new presentationPath
		mPath = newPath;		
	}
	
	/**
	 * Compares the object with another PresentationInfo object.
	 * 
	 * @param presentationInfo object to compare against
	 * @return <code>true</code> if the two objects are equivalent, 
	 * <code>false</code> otherwise
	 */
	public boolean equals(PresentationInfo presentationInfo)
	{
		if (presentationInfo == null)
		{
			return false;
		}
		
		return (mPath.toLowerCase().equals(presentationInfo.mPath.toLowerCase()));
	}
	
	/**
	 * Returns the presentation path.
	 */
	public String getPresentationPath()
	{
		return new String(mPath);
	}

	/**
	 * Returns the WAD filename for this presentation.
	 */
	public String getPAMFilename(String presentationHome)
	{
		int lastSeparator;
		String returnString = new String();
		boolean noHomePath = false;
		
		// Is the presentationHome null?  If so, assume that there is NO home path
		if (presentationHome == null)
		{
			noHomePath = true;
		}
		// Is the presentationHome empty?  If so, assume the current directory
		else if (presentationHome.equals(""))
		{
			returnString = new String(CURRENT_DIRECTORY);
		}
		else
		{
			returnString = new String(presentationHome);
		}		
		
		// Remove trailing file separator from the presentation home
		lastSeparator = returnString.lastIndexOf(File.separatorChar);
		if ((lastSeparator != -1) && (lastSeparator == (returnString.length() - 1)))
		{
			returnString = returnString.substring(0, returnString.length() - 1);
		}	
		
		if (noHomePath)
		{
			return (mPath + SPOTDefinitions.PAM_FILE_SUFFIX);
		}			
		return (returnString + File.separatorChar + mPath + SPOTDefinitions.PAM_FILE_SUFFIX);
	}

	/**
	 * Returns the presentation filename for this presentation.
	 */	
	public String getPresentationFilename(String presentationHome)
	{
		int lastSeparator;
		String returnString = new String();
		boolean noHomePath = false;
		
		// Is the presentationHome null?  If so, assume that there is NO home path
		if (presentationHome == null)
		{
			noHomePath = true;
		}
		// Is the presentationHome empty?  If so, assume the current directory
		else if (presentationHome.equals(""))
		{
			returnString = new String(CURRENT_DIRECTORY);
		}
		else
		{
			returnString = new String(presentationHome);
		}		
		
		// Remove trailing file separator from the presentation home
		lastSeparator = returnString.lastIndexOf(File.separatorChar);
		if ((lastSeparator != -1) && (lastSeparator == (returnString.length() - 1)))
		{
			returnString = returnString.substring(0, returnString.length() - 1);
		}	
		
		if (noHomePath)
		{
			return (mPath + SPOTDefinitions.PRESENTATION_FILE_SUFFIX);
		}		
		return (returnString + File.separatorChar + mPath + SPOTDefinitions.PRESENTATION_FILE_SUFFIX);
	}

	/**
	 * Returns the RAD filename for this presentation.
	 */	
	public String getRADFilename(String presentationHome)
	{
		int lastSeparator;
		String returnString = new String();
		boolean noHomePath = false;
		
		// Is the presentationHome null?  If so, assume that there is NO home path
		if (presentationHome == null)
		{
			noHomePath = true;
		}
		// Is the presentationHome empty?  If so, assume the current directory
		else if (presentationHome.equals(""))
		{
			returnString = new String(CURRENT_DIRECTORY);
		}
		else
		{
			returnString = new String(presentationHome);
		}
		
		// Remove trailing file separator from the presentation home
		lastSeparator = returnString.lastIndexOf(File.separatorChar);
		if ((lastSeparator != -1) && (lastSeparator == (returnString.length() - 1)))
		{
			returnString = returnString.substring(0, returnString.length() - 1);
		}	
		
		if (noHomePath)
		{
			return (mPath + SPOTDefinitions.RAD_FILE_SUFFIX);
		}
		return (returnString + File.separatorChar + mPath + SPOTDefinitions.RAD_FILE_SUFFIX);
	}
}